<?php
session_start();
require_once '../../api/db.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../login.php");
    exit();
}

// Get backup settings
$settings = [];
try {
    $result = $conn->query("SELECT setting_key, setting_value FROM settings WHERE setting_key LIKE 'backup_%'");
    while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
} catch (PDOException $e) {}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $keys = ['backup_enabled', 'backup_frequency', 'backup_retention_days', 'backup_auto_delete'];
    
    try {
        foreach ($keys as $key) {
            $value = $_POST[$key] ?? '';
            $stmt = $conn->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
            $stmt->execute([$key, $value, $value]);
        }
        $success = true;
    } catch (PDOException $e) {
        $error = true;
    }
}

// Get backup files
$backupDir = '../../backups/';
$backupFiles = [];
if (is_dir($backupDir)) {
    $files = scandir($backupDir, SCANDIR_SORT_DESCENDING);
    foreach ($files as $file) {
        if (str_ends_with($file, '.sql.gz')) {
            $backupFiles[] = [
                'name' => $file,
                'size' => filesize($backupDir . $file),
                'date' => filemtime($backupDir . $file)
            ];
        }
    }
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>پشتیبان‌گیری خودکار | پاکشو</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@100..900&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Vazirmatn', sans-serif; }
        .toggle-checkbox:checked {
            right: 0;
            border-color: #68D391;
        }
        .toggle-checkbox:checked + .toggle-label {
            background-color: #68D391;
        }
    </style>
</head>
<body class="bg-slate-50 text-gray-800 antialiased">
    <div class="min-h-screen flex flex-col">
        
        <!-- Header -->
        <header class="bg-white/80 backdrop-blur border-b border-gray-200 sticky top-0 z-50 px-6 py-4">
            <div class="max-w-5xl mx-auto flex items-center gap-5">
                <a href="settings_system.php" class="w-10 h-10 rounded-xl bg-gray-50 border border-gray-200 hover:bg-white hover:border-blue-300 hover:text-blue-600 hover:shadow-md transition-all duration-300 flex items-center justify-center text-gray-500">
                    <i class="fa-solid fa-arrow-right"></i>
                </a>
                <div>
                    <h1 class="text-xl font-bold text-gray-800">اتوماسیون بک‌آپ</h1>
                    <p class="text-xs text-gray-500 mt-0.5">مدیریت زمان‌بندی و آرشیو فایل‌های پشتیبان</p>
                </div>
            </div>
        </header>

        <main class="flex-1 p-6 md:p-8">
            <div class="max-w-5xl mx-auto space-y-8">
                
                <!-- Notifications -->
                <?php if (isset($success)): ?>
                <div class="p-4 bg-emerald-50 border border-emerald-200 rounded-xl text-emerald-700 flex items-center gap-3 animate-pulse">
                    <i class="fa-solid fa-circle-check text-xl"></i>
                    <span class="font-bold text-sm">تنظیمات با موفقیت ذخیره شد.</span>
                </div>
                <?php endif; ?>

                <?php if (isset($error)): ?>
                <div class="p-4 bg-rose-50 border border-rose-200 rounded-xl text-rose-700 flex items-center gap-3">
                    <i class="fa-solid fa-circle-exclamation text-xl"></i>
                    <span class="font-bold text-sm">خطا در ذخیره تنظیمات.</span>
                </div>
                <?php endif; ?>

                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    
                    <!-- Settings Column -->
                    <div class="lg:col-span-1 space-y-6">
                        <form method="POST" class="space-y-6">
                            
                            <!-- Main Config Card -->
                            <div class="bg-white rounded-3xl border border-gray-200 shadow-sm p-6 relative overflow-hidden">
                                <div class="absolute top-0 right-0 w-24 h-24 bg-blue-50 rounded-bl-full -mr-4 -mt-4 z-0"></div>
                                
                                <div class="relative z-10 space-y-6">
                                    <div class="flex items-center gap-3 mb-2">
                                        <div class="w-10 h-10 rounded-full bg-blue-100 text-blue-600 flex items-center justify-center">
                                            <i class="fa-solid fa-robot"></i>
                                        </div>
                                        <h2 class="font-bold text-gray-800">تنظیمات ربات</h2>
                                    </div>

                                    <!-- Enable Switch -->
                                    <div class="flex items-center justify-between p-4 bg-gray-50 rounded-2xl border border-gray-100">
                                        <span class="text-sm font-bold text-gray-700">وضعیت فعال</span>
                                        <label class="relative inline-flex items-center cursor-pointer">
                                            <input type="checkbox" name="backup_enabled" value="1" class="sr-only peer" <?php echo ($settings['backup_enabled'] ?? '') == '1' ? 'checked' : ''; ?>>
                                            <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                        </label>
                                    </div>

                                    <!-- Frequency -->
                                    <div>
                                        <label class="block text-xs font-bold text-gray-500 mb-2 mr-1">دوره زمانی پشتیبان‌گیری</label>
                                        <div class="relative">
                                            <select name="backup_frequency" class="w-full pl-4 pr-10 py-3 bg-white border border-gray-200 rounded-xl focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all appearance-none cursor-pointer text-sm">
                                                <option value="daily" <?php echo ($settings['backup_frequency'] ?? 'daily') == 'daily' ? 'selected' : ''; ?>>هر روز (Daily)</option>
                                                <option value="weekly" <?php echo ($settings['backup_frequency'] ?? '') == 'weekly' ? 'selected' : ''; ?>>هفتگی (Weekly)</option>
                                                <option value="monthly" <?php echo ($settings['backup_frequency'] ?? '') == 'monthly' ? 'selected' : ''; ?>>ماهانه (Monthly)</option>
                                            </select>
                                            <span class="absolute left-4 top-3.5 text-gray-400 pointer-events-none"><i class="fa-solid fa-chevron-down text-xs"></i></span>
                                            <span class="absolute right-4 top-3.5 text-blue-500"><i class="fa-solid fa-clock-rotate-left"></i></span>
                                        </div>
                                    </div>

                                    <!-- Retention -->
                                    <div>
                                        <label class="block text-xs font-bold text-gray-500 mb-2 mr-1">مدت نگهداری فایل‌ها (روز)</label>
                                        <div class="relative">
                                            <input type="number" name="backup_retention_days" value="<?php echo htmlspecialchars($settings['backup_retention_days'] ?? '30'); ?>" 
                                                min="7" step="1"
                                                class="w-full px-4 py-3 bg-white border border-gray-200 rounded-xl focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all text-sm font-bold text-center">
                                            <span class="absolute left-4 top-3.5 text-xs text-gray-400 font-bold">روز</span>
                                        </div>
                                    </div>

                                    <!-- Auto Delete Switch -->
                                    <div class="flex items-center gap-3 p-3 bg-rose-50 rounded-xl border border-rose-100">
                                        <input type="checkbox" name="backup_auto_delete" value="1" <?php echo ($settings['backup_auto_delete'] ?? '') == '1' ? 'checked' : ''; ?> 
                                            class="w-4 h-4 text-rose-600 rounded border-gray-300 focus:ring-rose-500 cursor-pointer">
                                        <div class="flex flex-col">
                                            <span class="text-xs font-bold text-rose-800">حذف خودکار قدیمی‌ها</span>
                                            <span class="text-[10px] text-rose-600/80">فایل‌های قدیمی‌تر از موعد حذف شوند</span>
                                        </div>
                                    </div>

                                    <button type="submit" class="w-full py-3 bg-gray-900 hover:bg-black text-white font-bold rounded-xl shadow-lg shadow-gray-400/20 transition-all transform active:scale-95 flex items-center justify-center gap-2">
                                        <i class="fa-solid fa-floppy-disk"></i>
                                        ذخیره تنظیمات
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>

                    <!-- Files List Column -->
                    <div class="lg:col-span-2">
                        <div class="bg-white rounded-3xl border border-gray-200 shadow-sm overflow-hidden h-full">
                            <div class="px-6 py-5 border-b border-gray-100 bg-gray-50 flex items-center justify-between">
                                <div class="flex items-center gap-3">
                                    <div class="w-10 h-10 rounded-full bg-purple-100 text-purple-600 flex items-center justify-center">
                                        <i class="fa-solid fa-server"></i>
                                    </div>
                                    <div>
                                        <h2 class="font-bold text-gray-800">آرشیو فایل‌ها</h2>
                                        <p class="text-xs text-gray-500">لیست فایل‌های موجود در سرور</p>
                                    </div>
                                </div>
                                <span class="text-xs font-bold bg-white border border-gray-200 px-3 py-1 rounded-full text-gray-600">
                                    <?php echo count($backupFiles); ?> فایل
                                </span>
                            </div>

                            <div class="p-4 max-h-[600px] overflow-y-auto custom-scrollbar">
                                <?php if (empty($backupFiles)): ?>
                                <div class="flex flex-col items-center justify-center py-16 text-gray-400">
                                    <div class="w-20 h-20 bg-gray-50 rounded-full flex items-center justify-center mb-4">
                                        <i class="fa-solid fa-box-open text-4xl opacity-50"></i>
                                    </div>
                                    <p class="font-bold text-sm">هنوز هیچ فایل پشتیبانی ایجاد نشده است.</p>
                                </div>
                                <?php else: ?>
                                <div class="grid grid-cols-1 gap-3">
                                    <?php foreach ($backupFiles as $file): ?>
                                    <div class="group flex items-center justify-between p-4 bg-white border border-gray-100 rounded-2xl hover:border-purple-200 hover:shadow-md hover:shadow-purple-100/50 transition-all duration-300">
                                        <div class="flex items-center gap-4">
                                            <div class="w-12 h-12 rounded-xl bg-purple-50 text-purple-600 flex items-center justify-center group-hover:scale-110 transition-transform">
                                                <i class="fa-solid fa-file-zipper text-xl"></i>
                                            </div>
                                            <div>
                                                <div class="font-bold text-gray-800 text-sm mb-1 dir-ltr text-right font-mono tracking-tight"><?php echo htmlspecialchars($file['name']); ?></div>
                                                <div class="flex items-center gap-3 text-xs text-gray-500">
                                                    <span class="flex items-center gap-1"><i class="fa-regular fa-calendar"></i> <?php echo date('Y/m/d H:i', $file['date']); ?></span>
                                                    <span class="w-1 h-1 bg-gray-300 rounded-full"></span>
                                                    <span class="flex items-center gap-1"><i class="fa-solid fa-weight-hanging"></i> <?php echo round($file['size'] / 1024 / 1024, 2); ?> MB</span>
                                                </div>
                                            </div>
                                        </div>
                                        <a href="../../backups/<?php echo urlencode($file['name']); ?>" download class="w-10 h-10 rounded-xl bg-gray-50 text-gray-600 hover:bg-blue-600 hover:text-white flex items-center justify-center transition-colors shadow-sm" title="دانلود فایل">
                                            <i class="fa-solid fa-cloud-arrow-down"></i>
                                        </a>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                </div>
            </div>
        </main>
    </div>
</body>
</html>