<?php
/**
 * Modern Installer Wizard - 2026 Edition
 * Standards: PHP 8.3+, TailwindCSS 4, Alpine.js
 */
declare(strict_types=1);

session_start();

// Security: CSRF Protection
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

class Installer {
    private const LOCK_FILE = 'includes/installed.lock';
    private const DB_CONFIG_FILE = 'includes/db_config.php';
    private const SQL_FILE = 'database.sql';

    public function __construct(
        private int $step = 1,
        private string $error = '',
        private string $success = ''
    ) {
        $this->step = isset($_GET['step']) ? max(1, (int)$_GET['step']) : 1;
        $this->checkInstallation();
    }

    private function checkInstallation(): void {
        if (file_exists(self::LOCK_FILE)) {
            header('Location: index.php');
            exit;
        }
    }

    public function handleRequest(): void {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') return;

        // CSRF Check
        if (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'] ?? '')) {
            $this->error = 'نشست شما منقضی شده است. لطفا صفحه را رفرش کنید.';
            return;
        }

        try {
            match ($this->step) {
                2 => $this->handleDatabaseCheck(),
                3 => $this->handleInstallation(),
                4 => $this->handleAdminCreation(),
                default => null
            };
        } catch (Throwable $e) {
            $this->error = $e->getMessage();
        }
    }

    private function handleDatabaseCheck(): void {
        $host = htmlspecialchars($_POST['db_host'] ?? '');
        $name = htmlspecialchars($_POST['db_name'] ?? '');
        $user = htmlspecialchars($_POST['db_user'] ?? '');
        $pass = $_POST['db_pass'] ?? '';

        $dsn = "mysql:host=$host;dbname=$name;charset=utf8mb4";
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_TIMEOUT => 5
        ];

        $conn = new PDO($dsn, $user, $pass, $options);
        $_SESSION['db_config'] = compact('host', 'name', 'user', 'pass');
        
        header('Location: ?step=3');
        exit;
    }

    private function handleInstallation(): void {
        $config = $_SESSION['db_config'] ?? throw new Exception('تنظیمات دیتابیس یافت نشد.');
        
        $dsn = "mysql:host={$config['host']};dbname={$config['name']};charset=utf8mb4";
        $conn = new PDO($dsn, $config['user'], $config['pass']);
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        if (!file_exists(self::SQL_FILE)) {
            throw new Exception('فایل دیتابیس (database.sql) یافت نشد.');
        }

        $sql = file_get_contents(self::SQL_FILE);
        // Execute multiple queries properly
        $conn->exec($sql);

        $configContent = sprintf(
            "<?php\ndeclare(strict_types=1);\n\ndefine('DB_HOST', '%s');\ndefine('DB_NAME', '%s');\ndefine('DB_USER', '%s');\ndefine('DB_PASS', '%s');\n",
            addslashes($config['host']),
            addslashes($config['name']),
            addslashes($config['user']),
            addslashes($config['pass'])
        );
        
        if (file_put_contents(self::DB_CONFIG_FILE, $configContent) === false) {
             throw new Exception('امکان نوشتن فایل کانفیگ وجود ندارد. لطفا دسترسی‌ها را بررسی کنید.');
        }

        header('Location: ?step=4');
        exit;
    }

    private function handleAdminCreation(): void {
        $name = htmlspecialchars($_POST['admin_name'] ?? '');
        $phone = htmlspecialchars($_POST['admin_phone'] ?? '');
        $pass = $_POST['admin_pass'] ?? '';

        if (!$name || !$phone || !$pass) {
            throw new Exception('تمامی فیلدها الزامی هستند.');
        }

        $config = $_SESSION['db_config'];
        $dsn = "mysql:host={$config['host']};dbname={$config['name']};charset=utf8mb4";
        $conn = new PDO($dsn, $config['user'], $config['pass']);
        
        // Modern password hashing
        $hashedPass = password_hash($pass, PASSWORD_ARGON2ID);

        $stmt = $conn->prepare("UPDATE users SET full_name=?, phone=?, password=? WHERE role='admin' LIMIT 1");
        // Note: Assuming 'users' table exists from step 3. If inserting new: "INSERT INTO users..."
        $stmt->execute([$name, $phone, $hashedPass]);

        file_put_contents(self::LOCK_FILE, date('c'));
        session_destroy(); // Cleanup setup session
        
        header('Location: ?step=5');
        exit;
    }

    public function getStep(): int { return $this->step; }
    public function getError(): string { return $this->error; }
}

$installer = new Installer();
$installer->handleRequest();
$step = $installer->getStep();
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl" class="scroll-smooth">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>نصب سیستم هوشمند | بلوار</title>
    
    <!-- Scripts & Styles -->
    <script src="https://cdn.tailwindcss.com"></script>
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.13.3/dist/cdn.min.js"></script>
    <link href="https://cdn.jsdelivr.net/gh/rastikerdar/vazirmatn@v33.003/Vazirmatn-font-face.css" rel="stylesheet" type="text/css" />
    
    <script>
        tailwind.config = {
            darkMode: 'media',
            theme: {
                extend: {
                    fontFamily: { sans: ['Vazirmatn', 'sans-serif'] },
                    colors: {
                        primary: { 50: '#eff6ff', 100: '#dbeafe', 500: '#3b82f6', 600: '#2563eb', 900: '#1e3a8a' },
                        glass: 'rgba(255, 255, 255, 0.7)',
                        glassDark: 'rgba(17, 24, 39, 0.7)'
                    },
                    backdropBlur: { 'xs': '2px' }
                }
            }
        }
    </script>
    <style>
        body {
            background-color: #f3f4f6;
            background-image: 
                radial-gradient(at 0% 0%, hsla(253,16%,7%,1) 0, transparent 50%), 
                radial-gradient(at 50% 0%, hsla(225,39%,30%,1) 0, transparent 50%), 
                radial-gradient(at 100% 0%, hsla(339,49%,30%,1) 0, transparent 50%);
            min-height: 100vh;
        }
        @media (prefers-color-scheme: light) {
            body {
                background-image: 
                    radial-gradient(at 40% 20%, hsla(28,100%,74%,1) 0px, transparent 50%),
                    radial-gradient(at 80% 0%, hsla(189,100%,56%,1) 0px, transparent 50%),
                    radial-gradient(at 0% 50%, hsla(355,100%,93%,1) 0px, transparent 50%);
            }
        }
        .glass-panel {
            background: var(--tw-colors-glass);
            backdrop-filter: blur(16px);
            -webkit-backdrop-filter: blur(16px);
            border: 1px solid rgba(255, 255, 255, 0.3);
        }
        @media (prefers-color-scheme: dark) {
            .glass-panel {
                background: var(--tw-colors-glassDark);
                border: 1px solid rgba(255, 255, 255, 0.1);
                color: #e5e7eb;
            }
        }
    </style>
</head>
<body class="flex items-center justify-center py-10 px-4">

    <div class="w-full max-w-5xl grid grid-cols-1 lg:grid-cols-12 gap-6" x-data="{ currentStep: <?= $step ?> }">
        
        <!-- Sidebar / Steps -->
        <div class="lg:col-span-4 space-y-4">
            <div class="glass-panel rounded-3xl p-8 shadow-2xl h-full flex flex-col justify-between relative overflow-hidden">
                <!-- Decorative Circle -->
                <div class="absolute -top-10 -right-10 w-32 h-32 bg-blue-500 rounded-full blur-3xl opacity-20"></div>

                <div>
                    <h1 class="text-3xl font-black mb-1 dark:text-white">بلوار</h1>
                    <p class="text-sm opacity-60 mb-8 dark:text-gray-300">نصب‌کننده هوشمند نسخه 2.0</p>
                    
                    <div class="space-y-6 relative">
                        <!-- Connecting Line -->
                        <div class="absolute right-[19px] top-4 bottom-4 w-0.5 bg-gray-200 dark:bg-gray-700 -z-10"></div>
                        
                        <?php 
                        $steps = [
                            1 => ['label' => 'خوش‌آمدید', 'desc' => 'شروع فرآیند'],
                            2 => ['label' => 'پایگاه داده', 'desc' => 'تنظیمات اتصال'],
                            3 => ['label' => 'نصب هسته', 'desc' => 'ساختار جداول'],
                            4 => ['label' => 'مدیریت', 'desc' => 'حساب کاربری'],
                            5 => ['label' => 'پایان', 'desc' => 'ورود به سیستم'],
                        ];
                        foreach($steps as $num => $info): 
                            $status = $num < $step ? 'done' : ($num == $step ? 'active' : 'wait');
                        ?>
                        <div class="flex items-center gap-4 group">
                            <div class="w-10 h-10 rounded-2xl flex items-center justify-center transition-all duration-500 shadow-lg
                                <?= $status === 'active' ? 'bg-blue-600 text-white scale-110 ring-4 ring-blue-500/20' : 
                                   ($status === 'done' ? 'bg-emerald-500 text-white' : 'bg-gray-200 dark:bg-gray-800 text-gray-400') ?>">
                                <?= $status === 'done' ? '✓' : $num ?>
                            </div>
                            <div>
                                <h3 class="font-bold text-sm transition-colors <?= $status === 'active' ? 'text-blue-600 dark:text-blue-400' : 'dark:text-gray-200' ?>">
                                    <?= $info['label'] ?>
                                </h3>
                                <p class="text-xs opacity-50 dark:text-gray-400"><?= $info['desc'] ?></p>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <div class="mt-8 text-xs opacity-40 text-center">
                    &copy; 2026 Bolvar System
                </div>
            </div>
        </div>

        <!-- Main Content -->
        <div class="lg:col-span-8">
            <div class="glass-panel rounded-3xl p-8 shadow-2xl min-h-[500px] flex flex-col justify-center relative">
                
                <?php if ($installer->getError()): ?>
                <div x-data="{show: true}" x-show="show" class="mb-6 p-4 rounded-xl bg-red-50/90 border border-red-200 text-red-700 flex items-center gap-3 backdrop-blur-sm animate-pulse">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                    <span class="font-medium"><?= htmlspecialchars($installer->getError()) ?></span>
                    <button @click="show = false" class="mr-auto opacity-50 hover:opacity-100">&times;</button>
                </div>
                <?php endif; ?>

                <!-- Step 1: Welcome -->
                <?php if ($step === 1): ?>
                <div class="text-center space-y-6" x-transition>
                    <div class="w-24 h-24 bg-gradient-to-tr from-blue-500 to-purple-600 rounded-[2rem] mx-auto flex items-center justify-center shadow-2xl shadow-blue-500/30 transform hover:rotate-6 transition-transform duration-500">
                        <svg class="w-12 h-12 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M13 10V3L4 14h7v7l9-11h-7z"></path></svg>
                    </div>
                    <h2 class="text-3xl font-black text-gray-800 dark:text-white tracking-tight">آماده پرواز هستید؟</h2>
                    <p class="text-gray-500 dark:text-gray-300 max-w-md mx-auto leading-relaxed">
                        نصب سیستم بلوار با متدولوژی جدید 2026 بازطراحی شده است.
                        امنیت بالاتر، سرعت بیشتر و رابط کاربری هوشمند.
                    </p>
                    
                    <div class="grid grid-cols-2 gap-4 max-w-sm mx-auto text-sm text-gray-500 dark:text-gray-400">
                        <div class="flex items-center gap-2 bg-white/50 dark:bg-black/20 p-3 rounded-xl">
                            <span class="w-2 h-2 rounded-full bg-green-500"></span> PHP 8.3+
                        </div>
                        <div class="flex items-center gap-2 bg-white/50 dark:bg-black/20 p-3 rounded-xl">
                            <span class="w-2 h-2 rounded-full bg-blue-500"></span> PDO MySQL
                        </div>
                    </div>

                    <div class="pt-4">
                        <a href="?step=2" class="inline-flex items-center justify-center px-8 py-4 bg-gray-900 dark:bg-white dark:text-gray-900 text-white rounded-2xl font-bold hover:scale-105 transition-all shadow-xl hover:shadow-2xl">
                            شروع نصب هوشمند
                            <svg class="w-5 h-5 mr-2 rtl:rotate-180" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8l4 4m0 0l-4 4m4-4H3"></path></svg>
                        </a>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Step 2: Database -->
                <?php if ($step === 2): ?>
                <div x-transition>
                    <div class="mb-6">
                        <h2 class="text-2xl font-bold dark:text-white">پیکربندی دیتابیس</h2>
                        <p class="text-gray-500 text-sm mt-1">اطلاعات اتصال به MySQL را وارد کنید.</p>
                    </div>
                    
                    <form method="POST" class="space-y-4">
                        <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div class="space-y-2">
                                <label class="text-sm font-medium dark:text-gray-300">آدرس سرور (Host)</label>
                                <input type="text" name="db_host" value="localhost" class="w-full px-4 py-3 rounded-xl bg-white/50 dark:bg-black/20 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all" required>
                            </div>
                            <div class="space-y-2">
                                <label class="text-sm font-medium dark:text-gray-300">نام دیتابیس</label>
                                <input type="text" name="db_name" value="khoshkshoei" class="w-full px-4 py-3 rounded-xl bg-white/50 dark:bg-black/20 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all" required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div class="space-y-2">
                                <label class="text-sm font-medium dark:text-gray-300">نام کاربری</label>
                                <input type="text" name="db_user" value="root" class="w-full px-4 py-3 rounded-xl bg-white/50 dark:bg-black/20 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all" required>
                            </div>
                            <div class="space-y-2">
                                <label class="text-sm font-medium dark:text-gray-300">رمز عبور</label>
                                <input type="password" name="db_pass" class="w-full px-4 py-3 rounded-xl bg-white/50 dark:bg-black/20 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all">
                            </div>
                        </div>

                        <div class="flex items-center justify-between pt-6">
                            <a href="?step=1" class="text-gray-500 hover:text-gray-800 dark:hover:text-white transition-colors">بازگشت</a>
                            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-8 py-3 rounded-xl font-bold shadow-lg shadow-blue-500/30 transition-all hover:scale-105">
                                بررسی اتصال
                            </button>
                        </div>
                    </form>
                </div>
                <?php endif; ?>

                <!-- Step 3: Installing -->
                <?php if ($step === 3): ?>
                <div class="text-center" x-transition>
                    <div class="relative w-32 h-32 mx-auto mb-6">
                        <div class="absolute inset-0 border-4 border-gray-200 dark:border-gray-700 rounded-full"></div>
                        <div class="absolute inset-0 border-4 border-blue-500 rounded-full border-t-transparent animate-spin"></div>
                        <div class="absolute inset-0 flex items-center justify-center">
                            <svg class="w-10 h-10 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"></path></svg>
                        </div>
                    </div>
                    
                    <h2 class="text-2xl font-bold mb-2 dark:text-white">ارتباط برقرار شد</h2>
                    <p class="text-gray-500 mb-8">با کلیک روی دکمه زیر، جداول دیتابیس ایجاد می‌شوند.</p>
                    
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
                        <button type="submit" class="w-full max-w-xs mx-auto bg-gradient-to-r from-blue-600 to-indigo-600 text-white py-4 rounded-xl font-bold shadow-xl hover:shadow-2xl hover:-translate-y-1 transition-all">
                            نصب پایگاه داده
                        </button>
                    </form>
                </div>
                <?php endif; ?>

                <!-- Step 4: Admin Info -->
                <?php if ($step === 4): ?>
                <div x-transition>
                    <div class="mb-6 text-center">
                        <h2 class="text-2xl font-bold dark:text-white">مدیر کل سیستم</h2>
                        <p class="text-gray-500 text-sm mt-1">این اطلاعات برای ورود به پنل مدیریت استفاده می‌شود.</p>
                    </div>

                    <form method="POST" class="space-y-5 max-w-md mx-auto">
                        <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
                        
                        <div class="relative">
                            <div class="absolute inset-y-0 right-0 flex items-center pr-4 pointer-events-none text-gray-400">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path></svg>
                            </div>
                            <input type="text" name="admin_name" placeholder="نام و نام خانوادگی" class="w-full pr-12 pl-4 py-4 rounded-xl bg-white/50 dark:bg-black/20 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all" required>
                        </div>

                        <div class="relative">
                            <div class="absolute inset-y-0 right-0 flex items-center pr-4 pointer-events-none text-gray-400">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z"></path></svg>
                            </div>
                            <input type="text" name="admin_phone" placeholder="شماره موبایل (نام کاربری)" class="w-full pr-12 pl-4 py-4 rounded-xl bg-white/50 dark:bg-black/20 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all" required>
                        </div>

                        <div class="relative">
                            <div class="absolute inset-y-0 right-0 flex items-center pr-4 pointer-events-none text-gray-400">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"></path></svg>
                            </div>
                            <input type="password" name="admin_pass" placeholder="رمز عبور قوی" class="w-full pr-12 pl-4 py-4 rounded-xl bg-white/50 dark:bg-black/20 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all" required>
                        </div>

                        <button type="submit" class="w-full bg-gray-900 dark:bg-white dark:text-gray-900 text-white py-4 rounded-xl font-bold text-lg hover:shadow-2xl transition-all hover:-translate-y-1">
                            تکمیل نصب
                        </button>
                    </form>
                </div>
                <?php endif; ?>

                <!-- Step 5: Finish -->
                <?php if ($step === 5): ?>
                <div class="text-center" x-transition>
                    <div class="w-28 h-28 bg-emerald-500 rounded-full flex items-center justify-center mx-auto mb-8 shadow-2xl shadow-emerald-500/40 animate-[bounce_2s_infinite]">
                        <svg class="w-14 h-14 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>
                    </div>
                    
                    <h2 class="text-3xl font-black text-emerald-600 dark:text-emerald-400 mb-4">نصب با موفقیت انجام شد!</h2>
                    <p class="text-gray-600 dark:text-gray-300 mb-8 leading-7">
                        فایل‌های امنیتی ایجاد شدند. برای امنیت بیشتر پوشه install را حذف کنید.<br>
                        اکنون می‌توانید از سیستم استفاده کنید.
                    </p>
                    
                    <div class="flex gap-4 justify-center">
                        <a href="index.php" class="px-8 py-3 bg-gray-200 dark:bg-gray-800 text-gray-700 dark:text-gray-200 rounded-xl font-bold hover:bg-gray-300 transition-colors">
                            مشاهده سایت
                        </a>
                        <a href="login.php" class="px-8 py-3 bg-blue-600 text-white rounded-xl font-bold shadow-lg hover:shadow-blue-500/40 hover:-translate-y-1 transition-all">
                            ورود به پنل مدیریت
                        </a>
                    </div>
                </div>
                <?php endif; ?>

            </div>
        </div>
    </div>

</body>
</html>
