<?php
session_start();
require_once '../../api/db.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../login.php");
    exit();
}

// Get users
$users = [];
try {
    $users = $conn->query("SELECT id, username, full_name, role, created_at FROM users ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {}

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' || (isset($_GET['action']) && $_GET['action'] === 'backup')) {
    $action = $_POST['action'] ?? $_GET['action'] ?? '';
    
    if ($action === 'change_password') {
        $old = $_POST['old_password'] ?? '';
        $new = $_POST['new_password'] ?? '';
        
        try {
            $stmt = $conn->prepare("SELECT password FROM users WHERE id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user && password_verify($old, $user['password'])) {
                $stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
                $stmt->execute([password_hash($new, PASSWORD_DEFAULT), $_SESSION['user_id']]);
                $success = "رمز عبور با موفقیت تغییر کرد.";
            } else {
                $error = "رمز عبور فعلی اشتباه است.";
            }
        } catch (PDOException $e) {
            $error = "خطا در تغییر رمز عبور.";
        }
    } elseif ($action === 'add_user') {
        try {
            $stmt = $conn->prepare("INSERT INTO users (username, password, full_name, role) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_POST['username'], password_hash($_POST['password'], PASSWORD_DEFAULT), $_POST['full_name'], $_POST['role']]);
            $success = "کاربر جدید با موفقیت اضافه شد.";
        } catch (PDOException $e) {
            $error = "خطا در افزودن کاربر (نام کاربری تکراری است).";
        }
    } elseif ($action === 'delete_user') {
        if ($_POST['id'] != $_SESSION['user_id']) {
            try {
                $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
                $stmt->execute([$_POST['id']]);
                $success = "کاربر با موفقیت حذف شد.";
            } catch (PDOException $e) {
                $error = "خطا در حذف کاربر.";
            }
        } else {
            $error = "شما نمی‌توانید حساب خودتان را حذف کنید.";
        }
    } elseif ($action === 'backup') {
        try {
            $backupType = $_GET['type'] ?? 'full';
            $tables = $backupType === 'orders' ? ['orders'] : ['users', 'orders', 'services'];
            $backup = "-- Backup Generated on " . date('Y-m-d H:i:s') . "\n\n";
            
            foreach ($tables as $table) {
                $result = $conn->query("SELECT * FROM $table");
                $rows = $result->fetchAll(PDO::FETCH_ASSOC);
                if ($rows) {
                    $backup .= "-- Table: $table\n";
                    $backup .= "DROP TABLE IF EXISTS `$table`;\n";
                    
                    $create = $conn->query("SHOW CREATE TABLE $table")->fetch(PDO::FETCH_ASSOC);
                    $backup .= $create['Create Table'] . ";\n\n";
                    
                    foreach ($rows as $row) {
                        $values = array_map(function($v) { return $v === null ? 'NULL' : "'" . addslashes($v) . "'"; }, $row);
                        $backup .= "INSERT INTO `$table` VALUES (" . implode(',', $values) . ");\n";
                    }
                    $backup .= "\n";
                }
            }
            
            // Compress with gzip
            $compressed = gzencode($backup, 9);
            
            header('Content-Type: application/gzip');
            header('Content-Disposition: attachment; filename="backup_' . $backupType . '_' . date('Y-m-d_H-i-s') . '.sql.gz"');
            header('Content-Length: ' . strlen($compressed));
            echo $compressed;
            exit();
        } catch (PDOException $e) {
            $error = "خطا در ایجاد فایل پشتیبان.";
        }
    } elseif ($action === 'restore') {
        if (isset($_FILES['backup_file']) && $_FILES['backup_file']['error'] === 0) {
            try {
                $file = $_FILES['backup_file']['tmp_name'];
                $filename = $_FILES['backup_file']['name'];
                
                // Check if gzip compressed
                if (str_ends_with($filename, '.gz')) {
                    $sql = gzdecode(file_get_contents($file));
                } else {
                    $sql = file_get_contents($file);
                }
                
                $statements = explode(';', $sql);
                
                $conn->beginTransaction();
                foreach ($statements as $statement) {
                    $statement = trim($statement);
                    if (!empty($statement) && !str_starts_with($statement, '--')) {
                        $conn->exec($statement);
                    }
                }
                $conn->commit();
                $success = "بازگردانی اطلاعات با موفقیت انجام شد.";
            } catch (Exception $e) {
                $conn->rollBack();
                $error = "خطا در بازگردانی: " . $e->getMessage();
            }
        } else {
            $error = "فایل پشتیبان انتخاب نشده است.";
        }
    } elseif ($action === 'delete_old') {
        $days = intval($_POST['days'] ?? 90);
        try {
            $stmt = $conn->prepare("DELETE FROM orders WHERE created_at < ? AND status = 'delivered'");
            $stmt->execute([time() - ($days * 86400)]);
            $success = "سفارشات قدیمی با موفقیت پاکسازی شدند.";
        } catch (PDOException $e) {
            $error = "خطا در حذف سفارشات.";
        }
    }
    
    // Refresh users list
    $users = $conn->query("SELECT id, username, full_name, role, created_at FROM users ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت سیستم | پاکشو</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.13.3/dist/cdn.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@100..900&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Vazirmatn', sans-serif; }
        [x-cloak] { display: none !important; }
        .custom-scrollbar::-webkit-scrollbar { width: 5px; height: 5px; }
        .custom-scrollbar::-webkit-scrollbar-track { background: #f1f5f9; }
        .custom-scrollbar::-webkit-scrollbar-thumb { background: #cbd5e1; border-radius: 5px; }
    </style>
</head>
<body class="bg-slate-50 text-gray-800 antialiased" x-data="systemSettings()">
    <div class="min-h-screen flex flex-col">
        
        <!-- Header -->
        <header class="bg-white/80 backdrop-blur-md border-b border-gray-200 sticky top-0 z-40 px-6 py-4">
            <div class="max-w-7xl mx-auto flex items-center justify-between">
                <div class="flex items-center gap-5">
                    <a href="settings.php" class="w-10 h-10 rounded-xl bg-gray-50 border border-gray-200 hover:bg-white hover:border-blue-300 hover:text-blue-600 hover:shadow-md transition-all duration-300 flex items-center justify-center text-gray-500">
                        <i class="fa-solid fa-arrow-right"></i>
                    </a>
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">تنظیمات پیشرفته سیستم</h1>
                        <p class="text-xs text-gray-500 mt-0.5">کاربران، امنیت و پایگاه داده</p>
                    </div>
                </div>
            </div>
        </header>

        <main class="flex-1 p-6 md:p-8">
            <div class="max-w-7xl mx-auto space-y-8">
                
                <!-- Alerts -->
                <div class="space-y-4">
                    <?php if (isset($success)): ?>
                    <div class="p-4 bg-emerald-50 border-r-4 border-emerald-500 rounded-lg shadow-sm flex items-center gap-3 animate-fade-in-down">
                        <i class="fa-solid fa-circle-check text-emerald-600 text-xl"></i>
                        <span class="text-sm font-bold text-emerald-800"><?php echo $success; ?></span>
                    </div>
                    <?php endif; ?>

                    <?php if (isset($error)): ?>
                    <div class="p-4 bg-rose-50 border-r-4 border-rose-500 rounded-lg shadow-sm flex items-center gap-3 animate-fade-in-down">
                        <i class="fa-solid fa-circle-exclamation text-rose-600 text-xl"></i>
                        <span class="text-sm font-bold text-rose-800"><?php echo $error; ?></span>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- Grid Layout -->
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    
                    <!-- Column 1 & 2: User Management -->
                    <div class="lg:col-span-2 space-y-6">
                        <div class="bg-white rounded-3xl border border-gray-200 shadow-sm overflow-hidden">
                            <div class="px-6 py-5 border-b border-gray-100 flex flex-col sm:flex-row items-center justify-between gap-4 bg-gradient-to-r from-gray-50 to-white">
                                <div class="flex items-center gap-3">
                                    <div class="w-10 h-10 rounded-full bg-blue-100 text-blue-600 flex items-center justify-center">
                                        <i class="fa-solid fa-users-gear"></i>
                                    </div>
                                    <h2 class="font-bold text-gray-800">مدیریت کاربران</h2>
                                </div>
                                <button @click="showUserModal = true; formData = {}" class="w-full sm:w-auto px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white text-sm font-bold rounded-xl transition shadow-lg shadow-blue-200 flex items-center justify-center gap-2">
                                    <i class="fa-solid fa-plus"></i>
                                    افزودن کاربر
                                </button>
                            </div>
                            
                            <div class="overflow-x-auto custom-scrollbar">
                                <table class="w-full">
                                    <thead class="bg-gray-50 text-gray-500 text-xs uppercase font-bold tracking-wider">
                                        <tr>
                                            <th class="px-6 py-4 text-right">کاربر</th>
                                            <th class="px-6 py-4 text-right">نام کاربری</th>
                                            <th class="px-6 py-4 text-center">نقش</th>
                                            <th class="px-6 py-4 text-center">عملیات</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-gray-100">
                                        <?php foreach ($users as $user): ?>
                                        <tr class="hover:bg-gray-50/50 transition-colors">
                                            <td class="px-6 py-4">
                                                <div class="flex items-center gap-3">
                                                    <div class="w-8 h-8 rounded-full bg-indigo-100 text-indigo-600 flex items-center justify-center text-xs font-bold">
                                                        <?php echo mb_substr($user['full_name'], 0, 1); ?>
                                                    </div>
                                                    <span class="font-bold text-gray-800 text-sm"><?php echo htmlspecialchars($user['full_name']); ?></span>
                                                </div>
                                            </td>
                                            <td class="px-6 py-4 text-sm text-gray-600 font-mono"><?php echo htmlspecialchars($user['username']); ?></td>
                                            <td class="px-6 py-4 text-center">
                                                <?php if ($user['role'] === 'admin'): ?>
                                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">
                                                        <i class="fa-solid fa-crown ml-1 text-[10px]"></i> مدیر
                                                    </span>
                                                <?php else: ?>
                                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                                                        <i class="fa-solid fa-user ml-1 text-[10px]"></i> کاربر
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="px-6 py-4 text-center">
                                                <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                                <form method="POST" class="inline" onsubmit="return confirm('آیا از حذف این کاربر اطمینان دارید؟')">
                                                    <input type="hidden" name="action" value="delete_user">
                                                    <input type="hidden" name="id" value="<?php echo $user['id']; ?>">
                                                    <button type="submit" class="w-8 h-8 rounded-lg text-rose-400 hover:bg-rose-50 hover:text-rose-600 transition flex items-center justify-center" title="حذف کاربر">
                                                        <i class="fa-solid fa-trash-can"></i>
                                                    </button>
                                                </form>
                                                <?php else: ?>
                                                    <span class="text-xs text-gray-300 italic">خودتان</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>

                        <!-- Maintenance Tools -->
                        <div class="bg-white rounded-3xl border border-gray-200 shadow-sm p-6">
                            <div class="flex items-center gap-3 mb-6">
                                <div class="w-10 h-10 rounded-full bg-amber-100 text-amber-600 flex items-center justify-center">
                                    <i class="fa-solid fa-screwdriver-wrench"></i>
                                </div>
                                <h2 class="font-bold text-gray-800">ابزارهای نگهداری</h2>
                            </div>
                            
                            <div class="grid grid-cols-2 sm:grid-cols-4 gap-4">
                                <button @click="showRestoreModal = true" class="flex flex-col items-center justify-center p-4 rounded-2xl bg-gray-50 border border-gray-200 hover:border-blue-400 hover:bg-blue-50 hover:text-blue-700 transition group">
                                    <i class="fa-solid fa-upload text-2xl mb-2 text-gray-400 group-hover:text-blue-600"></i>
                                    <span class="text-xs font-bold">بازگردانی بک‌آپ</span>
                                </button>
                                
                                <a href="settings_backup_auto.php" class="flex flex-col items-center justify-center p-4 rounded-2xl bg-gray-50 border border-gray-200 hover:border-purple-400 hover:bg-purple-50 hover:text-purple-700 transition group">
                                    <i class="fa-solid fa-robot text-2xl mb-2 text-gray-400 group-hover:text-purple-600"></i>
                                    <span class="text-xs font-bold">بک‌آپ خودکار</span>
                                </a>

                                <a href="fix_tracking_codes.php" class="flex flex-col items-center justify-center p-4 rounded-2xl bg-gray-50 border border-gray-200 hover:border-indigo-400 hover:bg-indigo-50 hover:text-indigo-700 transition group">
                                    <i class="fa-solid fa-wand-magic-sparkles text-2xl mb-2 text-gray-400 group-hover:text-indigo-600"></i>
                                    <span class="text-xs font-bold">اصلاح کدها</span>
                                </a>

                                <button @click="showDeleteModal = true" class="flex flex-col items-center justify-center p-4 rounded-2xl bg-gray-50 border border-gray-200 hover:border-rose-400 hover:bg-rose-50 hover:text-rose-700 transition group">
                                    <i class="fa-solid fa-broom text-2xl mb-2 text-gray-400 group-hover:text-rose-600"></i>
                                    <span class="text-xs font-bold">پاکسازی داده</span>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Column 3: Security & Database Backup -->
                    <div class="space-y-6">
                        
                        <!-- Security Card -->
                        <div class="bg-gradient-to-br from-gray-800 to-gray-900 rounded-3xl shadow-xl text-white p-6 relative overflow-hidden">
                            <div class="absolute top-0 right-0 p-4 opacity-10">
                                <i class="fa-solid fa-shield-halved text-8xl"></i>
                            </div>
                            <div class="relative z-10">
                                <h2 class="font-bold text-lg mb-1">امنیت حساب</h2>
                                <p class="text-gray-400 text-xs mb-6">آخرین ورود: هم‌اکنون</p>
                                
                                <button @click="showPasswordModal = true" class="w-full py-3 bg-white/10 hover:bg-white/20 border border-white/20 backdrop-blur-sm rounded-xl text-sm font-bold transition flex items-center justify-center gap-2">
                                    <i class="fa-solid fa-key"></i>
                                    تغییر رمز عبور
                                </button>
                            </div>
                        </div>

                        <!-- Backup Card -->
                        <div class="bg-white rounded-3xl border border-gray-200 shadow-sm p-6">
                            <div class="flex items-center gap-3 mb-4">
                                <div class="w-10 h-10 rounded-full bg-emerald-100 text-emerald-600 flex items-center justify-center">
                                    <i class="fa-solid fa-database"></i>
                                </div>
                                <h2 class="font-bold text-gray-800">پشتیبان‌گیری</h2>
                            </div>
                            
                            <div class="space-y-4">
                                <div class="p-3 bg-gray-50 rounded-xl space-y-2">
                                    <label class="flex items-center p-2 rounded-lg cursor-pointer hover:bg-white transition border border-transparent hover:border-gray-200">
                                        <input type="radio" name="backup_type" value="full" checked class="w-4 h-4 text-emerald-600 focus:ring-emerald-500 border-gray-300">
                                        <div class="mr-3">
                                            <span class="block text-sm font-bold text-gray-700">پشتیبان کامل</span>
                                            <span class="block text-[10px] text-gray-500">سفارشات + کاربران + تنظیمات</span>
                                        </div>
                                    </label>
                                    <label class="flex items-center p-2 rounded-lg cursor-pointer hover:bg-white transition border border-transparent hover:border-gray-200">
                                        <input type="radio" name="backup_type" value="orders" class="w-4 h-4 text-emerald-600 focus:ring-emerald-500 border-gray-300">
                                        <div class="mr-3">
                                            <span class="block text-sm font-bold text-gray-700">پشتیبان سریع</span>
                                            <span class="block text-[10px] text-gray-500">فقط جدول سفارشات</span>
                                        </div>
                                    </label>
                                </div>

                                <div x-show="downloadProgress > 0" x-cloak class="space-y-1">
                                    <div class="flex justify-between text-xs font-bold text-gray-600">
                                        <span>در حال ایجاد فایل...</span>
                                        <span x-text="Math.round(downloadProgress) + '%'"></span>
                                    </div>
                                    <div class="w-full bg-gray-100 rounded-full h-1.5 overflow-hidden">
                                        <div class="bg-emerald-500 h-1.5 rounded-full transition-all duration-300" :style="'width: ' + downloadProgress + '%'"></div>
                                    </div>
                                </div>

                                <button @click="downloadBackup()" class="w-full py-3 bg-emerald-600 hover:bg-emerald-700 text-white font-bold rounded-xl shadow-lg shadow-emerald-200 transition flex items-center justify-center gap-2">
                                    <i class="fa-solid fa-download"></i>
                                    دانلود فایل پشتیبان
                                </button>
                            </div>
                        </div>

                    </div>
                </div>
            </div>
        </main>

        <!-- Modals -->
        
        <!-- Add User Modal -->
        <div x-show="showUserModal" x-cloak class="fixed inset-0 z-50 flex items-center justify-center px-4">
            <div x-show="showUserModal" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100" x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100" x-transition:leave-end="opacity-0" class="absolute inset-0 bg-gray-900/60 backdrop-blur-sm" @click="showUserModal = false"></div>
            <div x-show="showUserModal" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 scale-90" x-transition:enter-end="opacity-100 scale-100" x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100 scale-100" x-transition:leave-end="opacity-0 scale-90" class="relative bg-white rounded-3xl shadow-2xl max-w-md w-full p-6" @click.stop>
                <h3 class="text-xl font-bold text-gray-800 mb-6 flex items-center gap-2">
                    <i class="fa-solid fa-user-plus text-blue-600"></i> کاربر جدید
                </h3>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="action" value="add_user">
                    <div>
                        <label class="block text-xs font-bold text-gray-500 mb-1">نام و نام خانوادگی</label>
                        <input type="text" name="full_name" required class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:bg-white focus:border-blue-500 transition outline-none text-sm">
                    </div>
                    <div>
                        <label class="block text-xs font-bold text-gray-500 mb-1">نام کاربری</label>
                        <input type="text" name="username" required class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:bg-white focus:border-blue-500 transition outline-none text-sm dir-ltr text-right">
                    </div>
                    <div>
                        <label class="block text-xs font-bold text-gray-500 mb-1">رمز عبور</label>
                        <input type="password" name="password" required class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:bg-white focus:border-blue-500 transition outline-none text-sm dir-ltr text-right">
                    </div>
                    <div>
                        <label class="block text-xs font-bold text-gray-500 mb-1">سطح دسترسی</label>
                        <select name="role" class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:bg-white focus:border-blue-500 transition outline-none text-sm">
                            <option value="user">کاربر عادی</option>
                            <option value="admin">مدیر کل</option>
                        </select>
                    </div>
                    <div class="pt-4 flex gap-3">
                        <button type="button" @click="showUserModal = false" class="flex-1 py-3 bg-gray-100 hover:bg-gray-200 text-gray-700 font-bold rounded-xl transition">لغو</button>
                        <button type="submit" class="flex-1 py-3 bg-blue-600 hover:bg-blue-700 text-white font-bold rounded-xl shadow-lg shadow-blue-200 transition">ذخیره</button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Password Modal -->
        <div x-show="showPasswordModal" x-cloak class="fixed inset-0 z-50 flex items-center justify-center px-4">
            <div x-show="showPasswordModal" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100" x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100" x-transition:leave-end="opacity-0" class="absolute inset-0 bg-gray-900/60 backdrop-blur-sm" @click="showPasswordModal = false"></div>
            <div x-show="showPasswordModal" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 scale-90" x-transition:enter-end="opacity-100 scale-100" x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100 scale-100" x-transition:leave-end="opacity-0 scale-90" class="relative bg-white rounded-3xl shadow-2xl max-w-md w-full p-6" @click.stop>
                <h3 class="text-xl font-bold text-gray-800 mb-6 flex items-center gap-2">
                    <i class="fa-solid fa-key text-gray-600"></i> تغییر رمز عبور
                </h3>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="action" value="change_password">
                    <div>
                        <label class="block text-xs font-bold text-gray-500 mb-1">رمز عبور فعلی</label>
                        <input type="password" name="old_password" required class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:bg-white focus:border-gray-800 transition outline-none text-sm dir-ltr text-right">
                    </div>
                    <div>
                        <label class="block text-xs font-bold text-gray-500 mb-1">رمز عبور جدید</label>
                        <input type="password" name="new_password" required class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:bg-white focus:border-gray-800 transition outline-none text-sm dir-ltr text-right">
                    </div>
                    <div class="pt-4 flex gap-3">
                        <button type="button" @click="showPasswordModal = false" class="flex-1 py-3 bg-gray-100 hover:bg-gray-200 text-gray-700 font-bold rounded-xl transition">انصراف</button>
                        <button type="submit" class="flex-1 py-3 bg-gray-900 hover:bg-black text-white font-bold rounded-xl shadow-lg shadow-gray-400 transition">تایید و تغییر</button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Restore Modal -->
        <div x-show="showRestoreModal" x-cloak class="fixed inset-0 z-50 flex items-center justify-center px-4">
            <div x-show="showRestoreModal" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100" x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100" x-transition:leave-end="opacity-0" class="absolute inset-0 bg-gray-900/60 backdrop-blur-sm" @click="showRestoreModal = false"></div>
            <div x-show="showRestoreModal" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 scale-90" x-transition:enter-end="opacity-100 scale-100" x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100 scale-100" x-transition:leave-end="opacity-0 scale-90" class="relative bg-white rounded-3xl shadow-2xl max-w-md w-full p-6" @click.stop>
                <div class="flex items-center justify-center w-12 h-12 rounded-full bg-blue-100 text-blue-600 mb-4 mx-auto">
                    <i class="fa-solid fa-cloud-arrow-up text-xl"></i>
                </div>
                <h3 class="text-xl font-bold text-gray-800 mb-2 text-center">بازگردانی اطلاعات</h3>
                <p class="text-sm text-gray-500 text-center mb-6">با بارگذاری فایل بک‌آپ، تمام اطلاعات فعلی سیستم حذف و با اطلاعات فایل جایگزین خواهد شد.</p>
                
                <form method="POST" enctype="multipart/form-data" class="space-y-4" onsubmit="return confirm('هشدار: اطلاعات فعلی حذف خواهند شد. آیا مطمئن هستید؟')">
                    <input type="hidden" name="action" value="restore">
                    <div class="relative group">
                        <input type="file" name="backup_file" accept=".sql,.gz" required class="absolute inset-0 w-full h-full opacity-0 cursor-pointer z-10">
                        <div class="w-full px-4 py-8 bg-gray-50 border-2 border-dashed border-gray-300 rounded-xl group-hover:border-blue-400 group-hover:bg-blue-50 transition flex flex-col items-center justify-center">
                            <i class="fa-solid fa-file-code text-gray-400 text-2xl mb-2 group-hover:text-blue-500"></i>
                            <span class="text-xs font-bold text-gray-500 group-hover:text-blue-600">برای انتخاب فایل کلیک کنید</span>
                            <span class="text-[10px] text-gray-400 mt-1">فرمت‌های مجاز: .sql یا .sql.gz</span>
                        </div>
                    </div>
                    
                    <div class="pt-2 flex gap-3">
                        <button type="button" @click="showRestoreModal = false" class="flex-1 py-3 bg-gray-100 hover:bg-gray-200 text-gray-700 font-bold rounded-xl transition">لغو</button>
                        <button type="submit" class="flex-1 py-3 bg-blue-600 hover:bg-blue-700 text-white font-bold rounded-xl shadow-lg shadow-blue-200 transition">شروع بازگردانی</button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Delete Old Modal -->
        <div x-show="showDeleteModal" x-cloak class="fixed inset-0 z-50 flex items-center justify-center px-4">
            <div x-show="showDeleteModal" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100" x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100" x-transition:leave-end="opacity-0" class="absolute inset-0 bg-gray-900/60 backdrop-blur-sm" @click="showDeleteModal = false"></div>
            <div x-show="showDeleteModal" x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 scale-90" x-transition:enter-end="opacity-100 scale-100" x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100 scale-100" x-transition:leave-end="opacity-0 scale-90" class="relative bg-white rounded-3xl shadow-2xl max-w-md w-full p-6" @click.stop>
                <div class="flex items-center justify-center w-12 h-12 rounded-full bg-rose-100 text-rose-600 mb-4 mx-auto">
                    <i class="fa-solid fa-trash-alt text-xl"></i>
                </div>
                <h3 class="text-xl font-bold text-gray-800 mb-2 text-center">پاکسازی سفارشات قدیمی</h3>
                <p class="text-sm text-gray-500 text-center mb-6">این عملیات سفارشات تکمیل شده‌ای که قدیمی‌تر از زمان مشخص شده باشند را برای همیشه حذف می‌کند.</p>
                
                <form method="POST" class="space-y-4" onsubmit="return confirm('آیا از حذف دائمی این اطلاعات اطمینان دارید؟ این عمل غیرقابل بازگشت است.')">
                    <input type="hidden" name="action" value="delete_old">
                    <div>
                        <label class="block text-xs font-bold text-gray-500 mb-1">حذف سفارشات قدیمی‌تر از:</label>
                        <div class="relative">
                            <input type="number" name="days" value="90" min="30" class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:bg-white focus:border-rose-500 transition outline-none text-sm text-center font-bold">
                            <span class="absolute left-4 top-3.5 text-xs text-gray-400 font-bold">روز</span>
                        </div>
                    </div>
                    
                    <div class="pt-2 flex gap-3">
                        <button type="button" @click="showDeleteModal = false" class="flex-1 py-3 bg-gray-100 hover:bg-gray-200 text-gray-700 font-bold rounded-xl transition">لغو</button>
                        <button type="submit" class="flex-1 py-3 bg-rose-600 hover:bg-rose-700 text-white font-bold rounded-xl shadow-lg shadow-rose-200 transition">پاکسازی</button>
                    </div>
                </form>
            </div>
        </div>

    </div>

    <script>
        function systemSettings() {
            return {
                showUserModal: false,
                showPasswordModal: false,
                showDeleteModal: false,
                showRestoreModal: false,
                formData: {},
                downloadProgress: 0,
                
                downloadBackup() {
                    const backupType = document.querySelector('input[name="backup_type"]:checked').value;
                    this.downloadProgress = 10;
                    
                    // Simulate progress
                    const interval = setInterval(() => {
                        if (this.downloadProgress < 90) {
                            this.downloadProgress += Math.random() * 15;
                        } else {
                            clearInterval(interval);
                        }
                    }, 200);
                    
                    // Trigger download
                    setTimeout(() => {
                        this.downloadProgress = 100;
                        const link = document.createElement('a');
                        link.href = '?action=backup&type=' + backupType;
                        link.click();
                        
                        // Reset after delay
                        setTimeout(() => {
                            this.downloadProgress = 0;
                        }, 1500);
                        clearInterval(interval);
                    }, 1500);
                }
            }
        }
    </script>
</body>
</html>