<?php
session_start();
header("Cache-Control: no-cache, no-store, must-revalidate");
require_once '../../api/db.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../login.php");
    exit();
}

try {
    $stmt = $conn->query("SELECT * FROM users WHERE role = 'user' ORDER BY created_at DESC");
    $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    $customers = [];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت مشتریان | پاکشو</title>

    <!-- Tailwind -->
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        sans: ['Vazirmatn', 'system-ui', 'sans-serif']
                    },
                    colors: {
                        primary: {
                            50:  '#eff6ff',
                            100: '#dbeafe',
                            500: '#3b82f6',
                            600: '#2563eb',
                            700: '#1d4ed8',
                        }
                    }
                }
            }
        }
    </script>

    <!-- Alpine v3 -->
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.14.0/dist/cdn.min.js"></script>

    <!-- Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">

    <!-- Font -->
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@100..900&display=swap" rel="stylesheet">

    <style>
        [x-cloak] { display: none !important; }
        body { font-family: 'Vazirmatn', system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; }
    </style>
</head>
<body class="bg-slate-50 text-slate-900" x-data="customersApp()" x-init="init()">

<div class="min-h-screen p-4 md:p-8">
    <div class="max-w-6xl mx-auto space-y-4 md:space-y-6">
        
        <!-- Header -->
        <header class="bg-white/90 backdrop-blur rounded-3xl shadow-sm border border-slate-100 px-4 py-4 md:px-6 md:py-5 flex flex-col md:flex-row md:items-center md:justify-between gap-4">
            <div class="flex items-center gap-3 md:gap-4">
                <a href="admin.php"
                   class="w-9 h-9 md:w-10 md:h-10 rounded-xl bg-slate-100 hover:bg-slate-200 text-slate-700 flex items-center justify-center transition-colors">
                    <i class="fa-solid fa-arrow-right text-sm"></i>
                </a>
                <div>
                    <div class="flex items-center gap-2">
                        <h1 class="text-xl md:text-2xl font-black tracking-tight">مدیریت مشتریان</h1>
                        <span class="px-2 py-0.5 text-[11px] rounded-full bg-primary-50 text-primary-600 border border-primary-100">
                            نسخه جدید
                        </span>
                    </div>
                    <p class="text-xs md:text-sm text-slate-500 mt-1">
                        مشاهده، جستجو و مدیریت مشتریان ثبت‌نام شده در سیستم
                    </p>
                </div>
            </div>
            <div class="flex items-center gap-3 md:gap-4">
                <div class="hidden sm:flex flex-col items-end text-xs text-slate-500">
                    <span>تعداد کل مشتریان</span>
                    <span class="text-base font-bold text-slate-800" x-text="customers.length.toLocaleString('fa-IR') + ' نفر'"></span>
                </div>
                <div class="flex items-center gap-2 bg-slate-900 text-white px-3 py-1.5 md:px-4 md:py-2 rounded-2xl text-xs md:text-sm">
                    <i class="fa-solid fa-users text-[13px]"></i>
                    <span x-text="filteredCustomers.length.toLocaleString('fa-IR') + ' نمایش داده شده'"></span>
                </div>
            </div>
        </header>

        <!-- Search & Filters -->
        <section class="bg-white rounded-3xl shadow-sm border border-slate-100 px-4 py-4 md:px-6 md:py-5 space-y-3 md:space-y-4">
            <div class="flex flex-col md:flex-row md:items-center gap-3 md:gap-4">
                <!-- Search box -->
                <div class="relative flex-1">
                    <i class="fa-solid fa-magnifying-glass absolute right-3 top-1/2 -translate-y-1/2 text-slate-400 text-xs md:text-sm"></i>
                    <input type="text"
                           x-model.debounce.250ms="search"
                           placeholder="جستجو در نام، شماره تماس یا آدرس مشتری..."
                           class="w-full h-10 md:h-11 rounded-xl border border-slate-200 bg-slate-50/70 pl-3 pr-9 text-sm focus:bg-white focus:border-primary-500 focus:ring-2 focus:ring-primary-100 outline-none transition-all">
                </div>

                <!-- Quick filters (در آینده می‌تونی بر اساس وضعیت، شهر، ... فیلتر کنی) -->
                <div class="flex items-center gap-2 text-xs md:text-[13px] text-slate-500">
                    <button type="button"
                            class="px-3 py-1.5 rounded-full border text-xs md:text-[13px]"
                            :class="activeFilter === 'all'
                                    ? 'border-slate-900 text-slate-900 bg-slate-900/5'
                                    : 'border-slate-200 hover:border-slate-400 hover:text-slate-800'"
                            @click="setFilter('all')">
                        همه
                    </button>
                    <button type="button"
                            class="px-3 py-1.5 rounded-full border text-xs md:text-[13px]"
                            :class="activeFilter === 'hasAddress'
                                    ? 'border-slate-900 text-slate-900 bg-slate-900/5'
                                    : 'border-slate-200 hover:border-slate-400 hover:text-slate-800'"
                            @click="setFilter('hasAddress')">
                        دارای آدرس
                    </button>
                    <button type="button"
                            class="px-3 py-1.5 rounded-full border text-xs md:text-[13px]"
                            :class="activeFilter === 'noAddress'
                                    ? 'border-slate-900 text-slate-900 bg-slate-900/5'
                                    : 'border-slate-200 hover:border-slate-400 hover:text-slate-800'"
                            @click="setFilter('noAddress')">
                        بدون آدرس
                    </button>
                </div>
            </div>

            <!-- Meta info -->
            <div class="flex flex-wrap items-center justify-between gap-2 text-[11px] md:text-xs text-slate-500">
                <div class="flex items-center gap-1.5">
                    <i class="fa-regular fa-circle-dot text-[10px] md:text-[11px]"></i>
                    <span>نتایج براساس آخرین تاریخ ثبت‌نام مرتب شده‌اند.</span>
                </div>
                <div class="flex items-center gap-2">
                    <span>مرتب‌سازی:</span>
                    <button type="button"
                            class="px-2 py-0.5 rounded-full border border-slate-200 hover:border-slate-400 flex items-center gap-1"
                            @click="toggleSortOrder">
                        <i :class="sortDesc ? 'fa-solid fa-arrow-down-wide-short' : 'fa-solid fa-arrow-up-short-wide'"></i>
                        <span x-text="sortDesc ? 'جدیدترین' : 'قدیمی‌ترین'"></span>
                    </button>
                </div>
            </div>
        </section>

        <!-- Customers Table -->
        <section class="bg-white rounded-3xl shadow-sm border border-slate-100 overflow-hidden">
            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-slate-50 border-b border-slate-200/80">
                        <tr>
                            <th class="px-5 py-3 text-right text-[11px] font-semibold text-slate-500 uppercase">مشتری</th>
                            <th class="px-5 py-3 text-right text-[11px] font-semibold text-slate-500 uppercase">شماره تماس</th>
                            <th class="px-5 py-3 text-right text-[11px] font-semibold text-slate-500 uppercase min-w-[200px]">آدرس</th>
                            <th class="px-5 py-3 text-center text-[11px] font-semibold text-slate-500 uppercase w-32">عملیات</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-slate-100" x-cloak>
                        <!-- Loading state -->
                        <tr x-show="isLoading">
                            <td colspan="4" class="px-6 py-16 text-center">
                                <div class="flex flex-col items-center gap-3">
                                    <div class="w-9 h-9 rounded-full border-2 border-primary-200 border-t-primary-600 animate-spin"></div>
                                    <p class="text-slate-500 text-sm">در حال بارگذاری مشتریان...</p>
                                </div>
                            </td>
                        </tr>

                        <!-- Customer rows -->
                        <template x-for="customer in filteredCustomers" :key="customer.id">
                            <tr class="hover:bg-slate-50/60 transition-colors" x-show="!isLoading">
                                <!-- نام -->
                                <td class="px-5 py-3.5">
                                    <div class="flex items-center gap-3">
                                        <div class="relative">
                                            <div class="w-9 h-9 rounded-full bg-gradient-to-br from-primary-500 to-purple-500 text-white flex items-center justify-center text-sm font-bold"
                                                 x-text="customer.full_name?.charAt(0) || '?'"></div>
                                        </div>
                                        <div class="space-y-0.5">
                                            <div class="font-semibold text-slate-900 text-sm"
                                                 x-text="customer.full_name || 'بدون نام'"></div>
                                            <div class="flex items-center gap-2 text-[11px] text-slate-500">
                                                <span class="px-1.5 py-0.5 rounded-full bg-slate-100 text-slate-600 border border-slate-200">
                                                    ID: <span x-text="customer.id"></span>
                                                </span>
                                                <span class="hidden sm:inline-block">
                                                    ساخت: 
                                                    <span x-text="formatDate(customer.created_at)"></span>
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </td>

                                <!-- تلفن -->
                                <td class="px-5 py-3.5 align-top">
                                    <template x-if="customer.phone">
                                        <a :href="'tel:' + customer.phone"
                                           class="text-slate-800 font-mono text-xs sm:text-sm hover:text-primary-600 hover:underline underline-offset-4"
                                           x-text="highlight(customer.phone)"></a>
                                    </template>
                                    <template x-if="!customer.phone">
                                        <span class="text-[11px] text-slate-400 italic">شماره ثبت نشده</span>
                                    </template>
                                </td>

                                <!-- آدرس -->
                                <td class="px-5 py-3.5">
                                    <template x-if="customer.address">
                                        <p class="text-xs sm:text-sm text-slate-700 leading-relaxed line-clamp-2"
                                           x-html="highlight(customer.address)"></p>
                                    </template>
                                    <template x-if="!customer.address">
                                        <span class="inline-flex items-center gap-1 text-[11px] text-amber-600 bg-amber-50 border border-amber-200 px-2 py-0.5 rounded-full">
                                            <i class="fa-solid fa-circle-exclamation text-[10px]"></i>
                                            آدرس ثبت نشده
                                        </span>
                                    </template>
                                </td>

                                <!-- عملیات -->
                                <td class="px-5 py-3.5">
                                    <div class="flex justify-center gap-2">
                                        <button type="button"
                                                @click="viewOrders(customer.id)"
                                                class="w-9 h-9 rounded-lg bg-primary-50 text-primary-700 hover:bg-primary-100 flex items-center justify-center transition-colors"
                                                title="مشاهده سفارش‌ها">
                                            <i class="fa-solid fa-list-ul text-[13px]"></i>
                                        </button>
                                        <button type="button"
                                                @click="openDeleteModal(customer)"
                                                class="w-9 h-9 rounded-lg bg-rose-50 text-rose-600 hover:bg-rose-100 flex items-center justify-center transition-colors"
                                                title="حذف مشتری">
                                            <i class="fa-solid fa-trash text-[13px]"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        </template>

                        <!-- Empty state -->
                        <tr x-show="!isLoading && filteredCustomers.length === 0">
                            <td colspan="4" class="px-6 py-16 text-center">
                                <div class="flex flex-col items-center gap-3">
                                    <div class="w-12 h-12 rounded-2xl bg-slate-100 flex items-center justify-center text-slate-400">
                                        <i class="fa-solid fa-users-slash text-xl"></i>
                                    </div>
                                    <p class="text-slate-600 text-sm font-medium">
                                        هیچ مشتری مطابق با جستجوی شما یافت نشد
                                    </p>
                                    <p class="text-xs text-slate-400">
                                        عبارت جستجو را تغییر دهید یا فیلترها را بررسی کنید.
                                    </p>
                                </div>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <!-- Footer summary -->
            <div class="flex flex-wrap items-center justify-between gap-2 px-4 py-3 border-t border-slate-100 text-[11px] md:text-xs text-slate-500">
                <div class="flex items-center gap-1.5">
                    <i class="fa-regular fa-rectangle-list text-[10px]"></i>
                    <span>نمایش <span class="font-semibold" x-text="filteredCustomers.length.toLocaleString('fa-IR')"></span> از
                        <span class="font-semibold" x-text="customers.length.toLocaleString('fa-IR')"></span> مشتری</span>
                </div>
            </div>
        </section>
    </div>
</div>

<!-- Delete confirmation modal -->
<div x-show="showDeleteModal"
     x-cloak
     class="fixed inset-0 z-40 flex items-center justify-center bg-black/30 backdrop-blur-sm"
     @keydown.escape.window="showDeleteModal = false">
    <div x-show="showDeleteModal"
         x-transition
         class="bg-white rounded-2xl shadow-xl max-w-sm w-full mx-4 p-5 border border-slate-100">
        <div class="flex items-start gap-3">
            <div class="w-9 h-9 rounded-xl bg-rose-50 text-rose-600 flex items-center justify-center">
                <i class="fa-solid fa-triangle-exclamation"></i>
            </div>
            <div class="space-y-1">
                <h2 class="text-sm font-bold text-slate-900">حذف مشتری</h2>
                <p class="text-xs text-slate-500 leading-relaxed">
                    آیا از حذف این مشتری مطمئن هستید؟
                    <span class="font-semibold text-slate-800" x-text="customerToDelete?.full_name"></span>
                    حذف این کاربر ممکن است روی گزارش‌ها اثر بگذارد.
                </p>
            </div>
        </div>
        <div class="flex justify-end gap-2 mt-4 pt-3 border-t border-slate-100">
            <button type="button"
                    class="px-3 py-1.5 rounded-lg border border-slate-200 text-xs text-slate-700 hover:bg-slate-50"
                    @click="showDeleteModal = false">
                انصراف
            </button>
            <button type="button"
                    class="px-3 py-1.5 rounded-lg bg-rose-600 text-xs text-white hover:bg-rose-700 disabled:opacity-60"
                    :disabled="isDeleting"
                    @click="confirmDelete">
                <span x-show="!isDeleting">حذف نهایی</span>
                <span x-show="isDeleting">در حال حذف...</span>
            </button>
        </div>
    </div>
</div>

<script>
function customersApp() {
    return {
        customers: <?php echo json_encode($customers, JSON_UNESCAPED_UNICODE); ?> || [],
        search: '',
        activeFilter: 'all',
        sortDesc: true,
        isLoading: true,
        showDeleteModal: false,
        customerToDelete: null,
        isDeleting: false,

        init() {
            // شبیه‌سازی لود برای حس بهتر UI؛ اگر نخواستی، همین خط را حذف کن.
            setTimeout(() => {
                this.isLoading = false;
            }, 200);
        },

        setFilter(filter) {
            this.activeFilter = filter;
        },

        toggleSortOrder() {
            this.sortDesc = !this.sortDesc;
            this.customers.reverse();
        },

        get filteredCustomers() {
            let list = [...this.customers];

            // فیلتر آدرس
            if (this.activeFilter === 'hasAddress') {
                list = list.filter(c => c.address && c.address.trim().length > 0);
            } else if (this.activeFilter === 'noAddress') {
                list = list.filter(c => !c.address || c.address.trim().length === 0);
            }

            // جستجو
            if (this.search.trim() !== '') {
                const q = this.search.toLowerCase();
                list = list.filter(c => {
                    const name = (c.full_name || '').toLowerCase();
                    const phone = (c.phone || '').toLowerCase();
                    const address = (c.address || '').toLowerCase();
                    return name.includes(q) || phone.includes(q) || address.includes(q);
                });
            }

            return list;
        },

        viewOrders(customerId) {
            window.location.href = `admin.php?customer_id=${customerId}`;
        },

        openDeleteModal(customer) {
            this.customerToDelete = customer;
            this.showDeleteModal = true;
        },

        async confirmDelete() {
            if (!this.customerToDelete) return;
            this.isDeleting = true;

            try {
                const res = await fetch(`../../api/users.php?id=${this.customerToDelete.id}`, {
                    method: 'DELETE',
                });

                if (res.ok) {
                    this.customers = this.customers.filter(c => c.id !== this.customerToDelete.id);
                    this.showDeleteModal = false;
                    this.customerToDelete = null;
                } else {
                    alert('خطا در حذف مشتری');
                }
            } catch (e) {
                alert('خطا در ارتباط با سرور');
            } finally {
                this.isDeleting = false;
            }
        },

        // هایلایت متن جستجو داخل رشته
        highlight(text) {
            if (!this.search.trim()) return this.escapeHtml(text);
            const q = this.search.trim().replace(/[.*+?^${}()|[\]\\]/g, '\\$&'); // escape
            const regex = new RegExp(q, 'gi');
            return this.escapeHtml(text).replace(regex, match => `<mark class="bg-yellow-200/70 text-inherit rounded px-0.5">${match}</mark>`);
        },

        escapeHtml(string) {
            if (!string) return '';
            return String(string)
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#039;');
        },

        // فرمت ساده تاریخ اگر نیاز بود (بر اساس created_at دیتابیس؛ می‌تونی با persian_date هم ترکیب کنی)
        formatDate(dt) {
            if (!dt) return '-';
            // اگر timestamp است
            if (!isNaN(dt)) {
                const d = new Date(Number(dt) * 1000);
                return d.toLocaleDateString('fa-IR');
            }
            // اگر رشته تاریخ/زمان است
            const d = new Date(dt);
            if (isNaN(d.getTime())) return dt;
            return d.toLocaleDateString('fa-IR');
        }
    }
}
</script>

</body>
</html>