// Utility Functions

// Base API URL (Relative for static hosting environment)
const API_BASE = 'api';

/**
 * Format number to Persian Currency format (e.g., 120,000 تومان)
 */
function formatCurrency(amount) {
    return new Intl.NumberFormat('fa-IR').format(amount) + ' تومان';
}

/**
 * Format Date to Persian (Jalali)
 */
function toPersianDate(dateString) {
    if (!dateString) return '-';
    // dateString usually comes as ms from table api created_at
    const date = new Date(Number(dateString));
    return new Intl.DateTimeFormat('fa-IR', {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    }).format(date);
}

/**
 * Generic Fetch Wrapper
 */
async function apiCall(endpoint, method = 'GET', data = null) {
    const options = {
        method,
        headers: {
            'Content-Type': 'application/json'
        }
    };
    if (data) {
        options.body = JSON.stringify(data);
    }

    try {
        const response = await fetch(endpoint, options);
        if (!response.ok) throw new Error('API Error');

        // Handle 204 No Content
        if (response.status === 204) return null;

        return await response.json();
    } catch (error) {
        console.error('API Call Failed:', error);
        return null;
    }
}

/**
 * Show Toast Notification (Simple Implementation)
 */
function showToast(message, type = 'success') {
    const toast = document.createElement('div');
    toast.className = `fixed bottom-4 left-4 px-6 py-3 rounded shadow-lg text-white z-50 transform transition-all duration-300 translate-y-full ${type === 'success' ? 'bg-green-600' : 'bg-red-600'}`;
    toast.innerText = message;
    document.body.appendChild(toast);

    // Animate in
    setTimeout(() => toast.classList.remove('translate-y-full'), 10);

    // Remove after 3s
    setTimeout(() => {
        toast.classList.add('translate-y-full');
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}
