<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET,POST,PUT,PATCH,DELETE,OPTIONS");
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') { http_response_code(200); exit(); }

include_once 'db.php';

$method = $_SERVER['REQUEST_METHOD'];

if ($method == 'GET') {
    // Check if phone parameter exists for single user lookup
    if (isset($_GET['phone'])) {
        $phone = $_GET['phone'];
        $stmt = $conn->prepare("SELECT full_name, phone, address FROM users WHERE phone = ?");
        $stmt->execute([$phone]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user) {
            echo json_encode($user);
        } else {
            http_response_code(404);
            echo json_encode(['message' => 'User not found']);
        }
        exit();
    }
    
    $search = isset($_GET['search']) ? $_GET['search'] : '';
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 100;

    $query = "SELECT * FROM users";
    $params = [];

    if (!empty($search)) {
        $query .= " WHERE phone LIKE :search OR full_name LIKE :search";
        $params[':search'] = "%$search%";
    }
    
    $query .= " ORDER BY id DESC LIMIT :limit";
    $params[':limit'] = $limit; // PDO limit binding issues sometimes, but direct query string for limit is okay if cast to int.
    
    // Better way for limit in PDO
    $stmt = $conn->prepare(str_replace(':limit', $limit, $query));
    if(!empty($search)){
        $stmt->bindParam(':search', $params[':search']);
    }
    
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // JS expects { data: [...] }
    echo json_encode(['data' => $users]);

} elseif ($method == 'POST') {
    // Create User
    $data = json_decode(file_get_contents("php://input"));

    if(isset($data->full_name) && isset($data->phone) && isset($data->password)) {
        
        // Check if phone exists (extra server-side check)
        $check = $conn->prepare("SELECT id FROM users WHERE phone = ?");
        $check->execute([$data->phone]);
        if($check->rowCount() > 0) {
            http_response_code(400);
            echo json_encode(["message" => "Phone already exists"]);
            exit();
        }

        $query = "INSERT INTO users (full_name, phone, password, address, role, created_at) VALUES (:name, :phone, :pass, :addr, :role, :created)";
        
        $stmt = $conn->prepare($query);
        
        // Use current time in ms
        $created_at = round(microtime(true) * 1000);
        
        $params = [
            ':name' => $data->full_name,
            ':phone' => $data->phone,
            ':pass' => password_hash($data->password, PASSWORD_DEFAULT),
            ':addr' => $data->address ?? '',
            ':role' => $data->role ?? 'user',
            ':created' => $created_at
        ];

        if($stmt->execute($params)) {
            $data->id = $conn->lastInsertId();
            $data->created_at = $created_at;
            // Return raw data (register.html logic: const resp = ...; login(resp))
            // Wait, register.html checks `if (resp)`. `apiCall` returns JSON.
            // If I return the user object, `login(resp)` works.
            // So return user object directly, OR {data: user}?
            // Auth logic expects `user.role`.
            // So return the user object directly.
            echo json_encode($data); 
        } else {
            http_response_code(503);
            echo json_encode(["message" => "Unable to create user"]);
        }
    } else {
        http_response_code(400);
        echo json_encode(["message" => "Incomplete data"]);
    }
}
?>
