<?php
session_start();
header('Content-Type: application/json; charset=utf-8');
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Pragma: no-cache");

require_once 'db.php';
require_once 'persian_date.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    http_response_code(401);
    exit(json_encode(['error' => 'Unauthorized']));
}

$period = $_GET['period'] ?? 'month';
$start_date = $_GET['start_date'] ?? null;
$end_date = $_GET['end_date'] ?? null;

$now = time();
switch ($period) {
    case 'today':
        $start = strtotime('today');
        $end = $now;
        break;
    case 'week':
        $start = strtotime('-7 days');
        $end = $now;
        break;
    case 'month':
        $start = strtotime('-30 days');
        $end = $now;
        break;
    case 'custom':
        $start = $start_date ? strtotime($start_date) : strtotime('-30 days');
        $end = $end_date ? strtotime($end_date) : $now;
        break;
    default:
        $start = strtotime('-30 days');
        $end = $now;
}

try {
    $stmt = $conn->prepare("SELECT * FROM orders WHERE created_at >= ? AND created_at <= ? ORDER BY created_at DESC");
    $stmt->execute([$start * 1000, $end * 1000]);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $stats = [
        'total_orders' => count($orders),
        'total_income' => 0,
        'completed' => 0,
        'pending' => 0,
        'cancelled' => 0,
        'avg_order' => 0
    ];
    
    $daily_income = [];
    foreach ($orders as &$order) {
        if ($order['status'] != 'cancelled') {
            $stats['total_income'] += $order['total_price'];
        }
        if ($order['status'] == 'delivered') $stats['completed']++;
        if ($order['status'] == 'pending') $stats['pending']++;
        if ($order['status'] == 'cancelled') $stats['cancelled']++;
        
        $day = date('Y-m-d', $order['created_at'] / 1000);
        if (!isset($daily_income[$day])) $daily_income[$day] = 0;
        if ($order['status'] != 'cancelled') {
            $daily_income[$day] += $order['total_price'];
        }
        
        // Add Persian date to order
        $order['created_at_persian'] = getPersianDate('Y/m/d', $order['created_at'] / 1000);
        $order['created_time_persian'] = getPersianDate('H:i', $order['created_at'] / 1000);
    }
    
    if ($stats['total_orders'] > 0) {
        $stats['avg_order'] = $stats['total_income'] / ($stats['total_orders'] - $stats['cancelled']);
    }
    
    // Sort dates and convert to Persian
    $sorted_dates = array_keys($daily_income);
    sort($sorted_dates);
    
    $persian_labels = [];
    $sorted_income = [];
    foreach ($sorted_dates as $date) {
        $timestamp = strtotime($date);
        $persian_labels[] = getPersianDate('Y/m/d', $timestamp);
        $sorted_income[] = $daily_income[$date];
    }
    
    echo json_encode([
        'stats' => $stats,
        'chart' => [
            'labels' => $persian_labels,
            'data' => $sorted_income
        ],
        'orders' => $orders
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error']);
}
