<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET,POST,PUT,PATCH,DELETE,OPTIONS");
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') { http_response_code(200); exit(); }

include_once 'db.php';
include_once 'sms.php';

// Function to send WhatsApp message
function sendWhatsAppMessage($phone, $message) {
    // Remove any non-numeric characters
    $phone = preg_replace('/\D/', '', $phone);
    
    // Ensure phone starts with country code
    if (!str_starts_with($phone, '98')) {
        if (str_starts_with($phone, '0')) {
            $phone = '98' . substr($phone, 1);
        } else {
            $phone = '98' . $phone;
        }
    }
    
    // Create WhatsApp link
    $whatsappUrl = 'https://api.whatsapp.com/send?phone=' . $phone . '&text=' . urlencode($message);
    
    // Log the message (optional)
    error_log("WhatsApp message queued for $phone");
    
    return true;
}

// Function to replace variables in message template
function replaceMessageVariables($template, $order) {
    $trackingCode = $order['tracking_code'] ?? '';
    
    // Log for debugging
    error_log("Original tracking code: " . $trackingCode);
    
    $replacements = [
        '{name}' => $order['customer_name'] ?? '',
        '{code}' => $trackingCode,
        '{price}' => number_format($order['total_price'] ?? 0),
        '{phone}' => $order['customer_phone'] ?? ''
    ];
    
    return str_replace(array_keys($replacements), array_values($replacements), $template);
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? $_GET['id'] : null;

if ($method == 'GET') {
    // Track order by tracking code and phone
    if (isset($_GET['action']) && $_GET['action'] == 'track') {
        $trackingCode = $_GET['tracking_code'] ?? '';
        $phone = $_GET['phone'] ?? '';
        
        if (empty($trackingCode) || empty($phone)) {
            echo json_encode(['error' => 'Missing parameters']);
            exit;
        }
        
        // Normalize tracking code - accept both formats (1404-123456 or 123456-1404)
        $trackingCodeAlt = $trackingCode;
        if (preg_match('/(\d+)-(\d+)/', $trackingCode, $matches)) {
            $trackingCodeAlt = $matches[2] . '-' . $matches[1]; // Swap format
        }
        
        $stmt = $conn->prepare("SELECT * FROM orders WHERE (tracking_code = ? OR tracking_code = ?) AND customer_phone = ?");
        $stmt->execute([$trackingCode, $trackingCodeAlt, $phone]);
        $order = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($order) {
            echo json_encode(['order' => $order]);
        } else {
            echo json_encode(['error' => 'Order not found']);
        }
        exit;
    }
    
    // List orders or get one
    if ($id) {
        $stmt = $conn->prepare("SELECT * FROM orders WHERE id = ?");
        $stmt->execute([$id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        echo json_encode($row);
    } else {
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 100;
        $orderBy = isset($_GET['sort']) ? $_GET['sort'] : 'created_at';
        $orderDir = isset($_GET['order']) && strtoupper($_GET['order']) == 'ASC' ? 'ASC' : 'DESC';

        // Whitelist columns for security
        $allowedSorts = ['id', 'created_at', 'total_price', 'status'];
        if (!in_array($orderBy, $allowedSorts)) $orderBy = 'created_at';

        $query = "SELECT * FROM orders ORDER BY $orderBy $orderDir LIMIT $limit";
        $stmt = $conn->prepare($query);
        $stmt->execute();
        $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Clean types
        foreach($orders as &$ord) {
            $ord['total_price'] = (float)$ord['total_price'];
            $ord['user_id'] = (int)$ord['user_id'];
            // items_json is string, that's fine
        }

        echo json_encode(['data' => $orders]);
    }

} elseif ($method == 'POST') {
    $data = json_decode(file_get_contents("php://input"));
    
    // Quick Customer Handling (Admin Panel)
    if (isset($data->customer_phone) && isset($data->customer_name)) {
        $phone = $data->customer_phone;
        $name = $data->customer_name;
        $address = $data->customer_address ?? '';
        
        // Check if user exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE phone = ?");
        $stmt->execute([$phone]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user) {
            $data->user_id = $user['id'];
            // Update address for existing user
            if (!empty($address)) {
                $upd = $conn->prepare("UPDATE users SET address = ? WHERE id = ?");
                $upd->execute([$address, $user['id']]);
            }
        } else {
            // Create new user with default password
            $defPass = password_hash('123456', PASSWORD_DEFAULT);
            $regStmt = $conn->prepare("INSERT INTO users (full_name, phone, password, role, created_at, address) VALUES (?, ?, ?, 'user', ?, ?)");
            $regStmt->execute([$name, $phone, $defPass, round(microtime(true) * 1000), $address]);
            $data->user_id = $conn->lastInsertId();
        }
    }

    if(isset($data->user_id) && isset($data->total_price)) {
        // ... proceeding with order creation using $data->user_id
        $query = "INSERT INTO orders (user_id, items_json, total_price, status, notes, tracking_code, customer_name, customer_phone, customer_address, delivery_date, created_at) VALUES (:uid, :items, :price, :status, :notes, :tracking, :cname, :cphone, :caddress, :delivery, :created)";
        $stmt = $conn->prepare($query);
        
        $params = [
            ':uid' => $data->user_id,
            ':items' => $data->items_json,
            ':price' => $data->total_price,
            ':status' => $data->status ?? 'pending',
            ':notes' => $data->notes ?? '',
            ':tracking' => $data->tracking_code ?? null,
            ':cname' => $data->customer_name ?? null,
            ':cphone' => $data->customer_phone ?? null,
            ':caddress' => $data->customer_address ?? null,
            ':delivery' => $data->delivery_date ?? null,
            ':created' => round(microtime(true) * 1000)
        ];

        if($stmt->execute($params)) {
             $data->id = $conn->lastInsertId();
             // Return tracking code clearly
             echo json_encode(['id' => $data->id, 'tracking_code' => $data->tracking_code ?? null, 'message' => "Order created"]);
        } else {
             http_response_code(500);
             echo json_encode(["message" => "Error creating order"]);
        }
    }
} elseif ($method == 'PUT' || $method == 'PATCH') {
    // Update status
    if (!$id) {
        // Try to get ID from URL if rewriting failed but passed as query string?
        // Or if the request was to /orders.php?id=1
        http_response_code(400);
        echo json_encode(["message" => "ID is required"]);
        exit;
    }

    $data = json_decode(file_get_contents("php://input"));
    
    // Build update query dynamically
    $fields = [];
    $params = [':id' => $id];
    
    if(isset($data->status)) {
        $fields[] = "status = :status";
        $params[':status'] = $data->status;
        
        // Send WhatsApp message on status change
        $stmt = $conn->prepare("SELECT * FROM orders WHERE id = ?");
        $stmt->execute([$id]);
        $order = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($order && $order['customer_phone']) {
            // Get message template from settings
            $msgKey = 'msg_' . $data->status;
            $settingStmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
            $settingStmt->execute([$msgKey]);
            $setting = $settingStmt->fetch(PDO::FETCH_ASSOC);
            
            if ($setting && $setting['setting_value']) {
                $message = replaceMessageVariables($setting['setting_value'], $order);
                
                // ارسال پیامک
                sendSMS($order['customer_phone'], $message);
            }
        }
    }
    if(isset($data->customer_name)) {
        $fields[] = "customer_name = :cname";
        $params[':cname'] = $data->customer_name;
    }
    if(isset($data->customer_phone)) {
        $fields[] = "customer_phone = :cphone";
        $params[':cphone'] = $data->customer_phone;
    }
    if(isset($data->customer_address)) {
        $fields[] = "customer_address = :caddress";
        $params[':caddress'] = $data->customer_address;
    }
    if(isset($data->delivery_date)) {
        $fields[] = "delivery_date = :delivery";
        $params[':delivery'] = $data->delivery_date;
    }
    if(isset($data->notes)) {
        $fields[] = "notes = :notes";
        $params[':notes'] = $data->notes;
    }
    if(isset($data->items_json)) {
        $fields[] = "items_json = :items";
        $params[':items'] = $data->items_json;
    }
    if(isset($data->total_price)) {
        $fields[] = "total_price = :price";
        $params[':price'] = $data->total_price;
    }

    if(empty($fields)) {
        echo json_encode(["message" => "No fields to update"]);
        exit;
    }

    $query = "UPDATE orders SET " . implode(', ', $fields) . " WHERE id = :id";
    $stmt = $conn->prepare($query);
    
    // Log the update for debugging
    error_log("Updating order $id with fields: " . implode(', ', array_keys($params)));
    
    if($stmt->execute($params)) {
        // Verify the update
        $verifyStmt = $conn->prepare("SELECT * FROM orders WHERE id = ?");
        $verifyStmt->execute([$id]);
        $updated = $verifyStmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode(["message" => "Order updated", "order" => $updated]);
    } else {
        error_log("Update failed: " . print_r($stmt->errorInfo(), true));
        http_response_code(500);
        echo json_encode(["message" => "Update failed", "error" => $stmt->errorInfo()]);
    }
} elseif ($method == 'DELETE') {
    if (!$id) {
        http_response_code(400);
        echo json_encode(["message" => "ID is required"]);
        exit;
    }
    
    $stmt = $conn->prepare("DELETE FROM orders WHERE id = ?");
    if($stmt->execute([$id])) {
        echo json_encode(["message" => "Order deleted"]);
    } else {
        http_response_code(500);
        echo json_encode(["message" => "Delete failed"]);
    }
}
?>
